/*
 * Decompiled with CFR 0.152.
 */
package org.apache.amoro.server.optimizing.maintainer;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.amoro.api.CommitMetaProducer;
import org.apache.amoro.config.DataExpirationConfig;
import org.apache.amoro.config.TableConfiguration;
import org.apache.amoro.io.AuthenticatedFileIO;
import org.apache.amoro.io.PathInfo;
import org.apache.amoro.io.SupportsFileSystemOperations;
import org.apache.amoro.server.optimizing.maintainer.AutoCreateIcebergTagAction;
import org.apache.amoro.server.optimizing.maintainer.TableMaintainer;
import org.apache.amoro.server.table.TableConfigurations;
import org.apache.amoro.server.table.TableOrphanFilesCleaningMetrics;
import org.apache.amoro.server.table.TableRuntime;
import org.apache.amoro.server.utils.IcebergTableUtil;
import org.apache.amoro.shade.guava32.com.google.common.annotations.VisibleForTesting;
import org.apache.amoro.shade.guava32.com.google.common.base.Predicate;
import org.apache.amoro.shade.guava32.com.google.common.base.Strings;
import org.apache.amoro.shade.guava32.com.google.common.collect.Iterables;
import org.apache.amoro.shade.guava32.com.google.common.collect.Maps;
import org.apache.amoro.shade.guava32.com.google.common.collect.Sets;
import org.apache.amoro.shade.guava32.com.google.common.primitives.Longs;
import org.apache.amoro.utils.TableFileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.iceberg.ContentFile;
import org.apache.iceberg.ContentScanTask;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.DeleteFile;
import org.apache.iceberg.DeleteFiles;
import org.apache.iceberg.FileContent;
import org.apache.iceberg.PartitionField;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.ReachableFileUtil;
import org.apache.iceberg.RewriteFiles;
import org.apache.iceberg.Schema;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.Table;
import org.apache.iceberg.TableScan;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.expressions.Literal;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.io.FileInfo;
import org.apache.iceberg.io.SupportsPrefixOperations;
import org.apache.iceberg.types.Conversions;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.ThreadPools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IcebergTableMaintainer
implements TableMaintainer {
    private static final Logger LOG = LoggerFactory.getLogger(IcebergTableMaintainer.class);
    public static final String METADATA_FOLDER_NAME = "metadata";
    public static final String DATA_FOLDER_NAME = "data";
    public static final String FLINK_JOB_ID = "flink.job-id";
    public static final String FLINK_MAX_COMMITTED_CHECKPOINT_ID = "flink.max-committed-checkpoint-id";
    public static final String EXPIRE_TIMESTAMP_MS = "TIMESTAMP_MS";
    public static final String EXPIRE_TIMESTAMP_S = "TIMESTAMP_S";
    public static final Set<String> AMORO_MAINTAIN_COMMITS = Sets.newHashSet((Object[])new String[]{CommitMetaProducer.OPTIMIZE.name(), CommitMetaProducer.DATA_EXPIRATION.name(), CommitMetaProducer.CLEAN_DANGLING_DELETE.name()});
    protected Table table;

    public IcebergTableMaintainer(Table table) {
        this.table = table;
    }

    @Override
    public void cleanOrphanFiles(TableRuntime tableRuntime) {
        TableConfiguration tableConfiguration = tableRuntime.getTableConfiguration();
        TableOrphanFilesCleaningMetrics orphanFilesCleaningMetrics = tableRuntime.getOrphanFilesCleaningMetrics();
        if (!tableConfiguration.isCleanOrphanEnabled()) {
            return;
        }
        long keepTime = tableConfiguration.getOrphanExistingMinutes() * 60L * 1000L;
        this.cleanContentFiles(System.currentTimeMillis() - keepTime, orphanFilesCleaningMetrics);
        this.table.refresh();
        this.cleanMetadata(System.currentTimeMillis() - keepTime, orphanFilesCleaningMetrics);
    }

    @Override
    public void cleanDanglingDeleteFiles(TableRuntime tableRuntime) {
        TableConfiguration tableConfiguration = tableRuntime.getTableConfiguration();
        if (!tableConfiguration.isDeleteDanglingDeleteFilesEnabled()) {
            return;
        }
        Snapshot currentSnapshot = this.table.currentSnapshot();
        if (currentSnapshot == null) {
            return;
        }
        Optional totalDeleteFiles = Optional.ofNullable(currentSnapshot.summary().get("total-delete-files"));
        if (totalDeleteFiles.isPresent() && Long.parseLong((String)totalDeleteFiles.get()) > 0L) {
            this.cleanDanglingDeleteFiles();
        } else {
            LOG.debug("There are no delete files here, so there is no need to clean dangling delete file for table {}", (Object)this.table.name());
        }
    }

    @Override
    public void expireSnapshots(TableRuntime tableRuntime) {
        if (!this.expireSnapshotEnabled(tableRuntime)) {
            return;
        }
        this.expireSnapshots(this.mustOlderThan(tableRuntime, System.currentTimeMillis()), tableRuntime.getTableConfiguration().getSnapshotMinCount());
    }

    protected boolean expireSnapshotEnabled(TableRuntime tableRuntime) {
        TableConfiguration tableConfiguration = tableRuntime.getTableConfiguration();
        return tableConfiguration.isExpireSnapshotEnabled();
    }

    @VisibleForTesting
    void expireSnapshots(long mustOlderThan, int minCount) {
        this.expireSnapshots(mustOlderThan, minCount, this.expireSnapshotNeedToExcludeFiles());
    }

    private void expireSnapshots(long olderThan, int minCount, Set<String> exclude) {
        LOG.debug("Starting snapshots expiration for table {}, expiring snapshots older than {} and retain last {} snapshots, excluding {}", new Object[]{this.table.name(), olderThan, minCount, exclude});
        AtomicInteger toDeleteFiles = new AtomicInteger(0);
        HashSet parentDirectories = new HashSet();
        HashSet expiredFiles = new HashSet();
        this.table.expireSnapshots().retainLast(Math.max(minCount, 1)).expireOlderThan(olderThan).deleteWith(file -> {
            if (exclude.isEmpty()) {
                expiredFiles.add(file);
            } else {
                String fileUriPath = TableFileUtil.getUriPath((String)file);
                if (!exclude.contains(fileUriPath) && !exclude.contains(new Path(fileUriPath).getParent().toString())) {
                    expiredFiles.add(file);
                }
            }
            parentDirectories.add(new Path(file).getParent().toString());
            toDeleteFiles.incrementAndGet();
        }).cleanExpiredFiles(true).commit();
        int deletedFiles = TableFileUtil.parallelDeleteFiles((AuthenticatedFileIO)this.fileIO(), expiredFiles, (ExecutorService)ThreadPools.getWorkerPool());
        parentDirectories.forEach(parent -> {
            try {
                TableFileUtil.deleteEmptyDirectory((AuthenticatedFileIO)this.fileIO(), (String)parent, (Set)exclude);
            }
            catch (Exception e) {
                LOG.warn("Failed to delete empty directory {} for table {}", new Object[]{parent, this.table.name(), e});
            }
        });
        this.runWithCondition(toDeleteFiles.get() > 0, () -> LOG.info("Deleted {}/{} files for table {}", new Object[]{deletedFiles, toDeleteFiles.get(), this.getTable().name()}));
    }

    @Override
    public void expireData(TableRuntime tableRuntime) {
        try {
            DataExpirationConfig expirationConfig = tableRuntime.getTableConfiguration().getExpiringDataConfig();
            Types.NestedField field = this.table.schema().findField(expirationConfig.getExpirationField());
            if (!TableConfigurations.isValidDataExpirationField(expirationConfig, field, this.table.name())) {
                return;
            }
            this.expireDataFrom(expirationConfig, this.expireBaseOnRule(expirationConfig, field));
        }
        catch (Throwable t) {
            LOG.error("Unexpected purge error for table {} ", (Object)tableRuntime.getTableIdentifier(), (Object)t);
        }
    }

    protected Instant expireBaseOnRule(DataExpirationConfig expirationConfig, Types.NestedField field) {
        switch (expirationConfig.getBaseOnRule()) {
            case CURRENT_TIME: {
                return Instant.now();
            }
            case LAST_COMMIT_TIME: {
                long lastCommitTimestamp = IcebergTableMaintainer.fetchLatestNonOptimizedSnapshotTime(this.getTable());
                if (lastCommitTimestamp != Long.MAX_VALUE) {
                    return Instant.ofEpochMilli(lastCommitTimestamp);
                }
                return Instant.MIN;
            }
        }
        throw new IllegalArgumentException("Cannot expire data base on " + expirationConfig.getBaseOnRule().name());
    }

    @VisibleForTesting
    public void expireDataFrom(DataExpirationConfig expirationConfig, Instant instant) {
        if (instant.equals(Instant.MIN)) {
            return;
        }
        long expireTimestamp = instant.minusMillis(expirationConfig.getRetentionTime()).toEpochMilli();
        LOG.info("Expiring data older than {} in table {} ", (Object)Instant.ofEpochMilli(expireTimestamp).atZone(IcebergTableMaintainer.getDefaultZoneId(this.table.schema().findField(expirationConfig.getExpirationField()))).toLocalDateTime(), (Object)this.table.name());
        Expression dataFilter = IcebergTableMaintainer.getDataExpression(this.table.schema(), expirationConfig, expireTimestamp);
        ExpireFiles expiredFiles = this.expiredFileScan(expirationConfig, dataFilter, expireTimestamp);
        this.expireFiles(expiredFiles, expireTimestamp);
    }

    @Override
    public void autoCreateTags(TableRuntime tableRuntime) {
        new AutoCreateIcebergTagAction(this.table, tableRuntime.getTableConfiguration().getTagConfiguration(), LocalDateTime.now()).execute();
    }

    protected void cleanContentFiles(long lastTime, TableOrphanFilesCleaningMetrics orphanFilesCleaningMetrics) {
        Set<String> validFiles = this.orphanFileCleanNeedToExcludeFiles();
        LOG.info("Starting cleaning orphan content files for table {}", (Object)this.table.name());
        this.clearInternalTableContentsFiles(lastTime, validFiles, orphanFilesCleaningMetrics);
    }

    protected void cleanMetadata(long lastTime, TableOrphanFilesCleaningMetrics orphanFilesCleaningMetrics) {
        LOG.info("Starting cleaning metadata files for table {}", (Object)this.table.name());
        this.clearInternalTableMetadata(lastTime, orphanFilesCleaningMetrics);
    }

    protected void cleanDanglingDeleteFiles() {
        LOG.info("Starting cleaning dangling delete files for table {}", (Object)this.table.name());
        int danglingDeleteFilesCnt = this.clearInternalTableDanglingDeleteFiles();
        this.runWithCondition(danglingDeleteFilesCnt > 0, () -> LOG.info("Deleted {} dangling delete files for table {}", (Object)danglingDeleteFilesCnt, (Object)this.table.name()));
    }

    protected long mustOlderThan(TableRuntime tableRuntime, long now) {
        return Longs.min((long[])new long[]{now - this.snapshotsKeepTime(tableRuntime), IcebergTableMaintainer.fetchOptimizingPlanSnapshotTime(this.table, tableRuntime), IcebergTableMaintainer.fetchLatestNonOptimizedSnapshotTime(this.table), IcebergTableMaintainer.fetchLatestFlinkCommittedSnapshotTime(this.table)});
    }

    protected long snapshotsKeepTime(TableRuntime tableRuntime) {
        return tableRuntime.getTableConfiguration().getSnapshotTTLMinutes() * 60L * 1000L;
    }

    protected Set<String> expireSnapshotNeedToExcludeFiles() {
        return Collections.emptySet();
    }

    protected Set<String> orphanFileCleanNeedToExcludeFiles() {
        return Sets.union(IcebergTableUtil.getAllContentFilePath(this.table), IcebergTableUtil.getAllStatisticsFilePath(this.table));
    }

    protected AuthenticatedFileIO fileIO() {
        return (AuthenticatedFileIO)this.table.io();
    }

    private void clearInternalTableContentsFiles(long lastTime, Set<String> exclude, TableOrphanFilesCleaningMetrics orphanFilesCleaningMetrics) {
        String dataLocation = this.table.location() + File.separator + DATA_FOLDER_NAME;
        int expected = 0;
        int deleted = 0;
        try (AuthenticatedFileIO io = this.fileIO();){
            if (io.supportFileSystemOperations()) {
                SupportsFileSystemOperations fio = io.asFileSystemIO();
                HashSet<PathInfo> directories = new HashSet<PathInfo>();
                Set<String> filesToDelete = this.deleteInvalidFilesInFs(fio, dataLocation, lastTime, exclude, directories);
                expected = filesToDelete.size();
                deleted = TableFileUtil.deleteFiles((AuthenticatedFileIO)io, filesToDelete);
                this.deleteEmptyDirectories(fio, directories, lastTime, exclude);
            } else if (io.supportPrefixOperations()) {
                SupportsPrefixOperations pio = io.asPrefixFileIO();
                Set<String> filesToDelete = this.deleteInvalidFilesByPrefix(pio, dataLocation, lastTime, exclude);
                expected = filesToDelete.size();
                deleted = TableFileUtil.deleteFiles((AuthenticatedFileIO)io, filesToDelete);
            } else {
                LOG.warn(String.format("Table %s doesn't support a fileIo with listDirectory or listPrefix, so skip clear files.", this.table.name()));
            }
        }
        int finalExpected = expected;
        int finalDeleted = deleted;
        this.runWithCondition(expected > 0, () -> {
            LOG.info("Deleted {}/{} orphan content files for table {}", new Object[]{finalDeleted, finalExpected, this.table.name()});
            orphanFilesCleaningMetrics.completeOrphanDataFiles(finalExpected, finalDeleted);
        });
    }

    private void clearInternalTableMetadata(long lastTime, TableOrphanFilesCleaningMetrics orphanFilesCleaningMetrics) {
        Set<String> validFiles = IcebergTableMaintainer.getValidMetadataFiles(this.table);
        LOG.info("Found {} valid metadata files for table {}", (Object)validFiles.size(), (Object)this.table.name());
        Pattern excludeFileNameRegex = IcebergTableMaintainer.getExcludeFileNameRegex(this.table);
        LOG.info("Exclude metadata files with name pattern {} for table {}", (Object)excludeFileNameRegex, (Object)this.table.name());
        String metadataLocation = this.table.location() + File.separator + METADATA_FOLDER_NAME;
        LOG.info("start orphan files clean in {}", (Object)metadataLocation);
        try (AuthenticatedFileIO io = this.fileIO();){
            if (io.supportPrefixOperations()) {
                SupportsPrefixOperations pio = io.asPrefixFileIO();
                Set<String> filesToDelete = this.deleteInvalidMetadataFile(pio, metadataLocation, lastTime, validFiles, excludeFileNameRegex);
                int deleted = TableFileUtil.deleteFiles((AuthenticatedFileIO)io, filesToDelete);
                this.runWithCondition(!filesToDelete.isEmpty(), () -> {
                    LOG.info("Deleted {}/{} metadata files for table {}", new Object[]{deleted, filesToDelete.size(), this.table.name()});
                    orphanFilesCleaningMetrics.completeOrphanMetadataFiles(filesToDelete.size(), deleted);
                });
            } else {
                LOG.warn(String.format("Table %s doesn't support a fileIo with listDirectory or listPrefix, so skip clear files.", this.table.name()));
            }
        }
    }

    private int clearInternalTableDanglingDeleteFiles() {
        Set<DeleteFile> danglingDeleteFiles = IcebergTableUtil.getDanglingDeleteFiles(this.table);
        if (danglingDeleteFiles.isEmpty()) {
            return 0;
        }
        RewriteFiles rewriteFiles = this.table.newRewrite();
        rewriteFiles.rewriteFiles(Collections.emptySet(), danglingDeleteFiles, Collections.emptySet(), Collections.emptySet());
        try {
            rewriteFiles.set("snapshot.producer", CommitMetaProducer.CLEAN_DANGLING_DELETE.name());
            rewriteFiles.commit();
        }
        catch (ValidationException e) {
            LOG.warn("Failed to commit dangling delete file for table {}, but ignore it", (Object)this.table.name(), (Object)e);
            return 0;
        }
        return danglingDeleteFiles.size();
    }

    public static long fetchLatestFlinkCommittedSnapshotTime(Table table) {
        Snapshot snapshot = IcebergTableMaintainer.findLatestSnapshotContainsKey(table, FLINK_MAX_COMMITTED_CHECKPOINT_ID);
        return snapshot == null ? Long.MAX_VALUE : snapshot.timestampMillis();
    }

    public static long fetchOptimizingPlanSnapshotTime(Table table, TableRuntime tableRuntime) {
        if (tableRuntime.getOptimizingStatus().isProcessing()) {
            long fromSnapshotId = tableRuntime.getOptimizingProcess().getTargetSnapshotId();
            for (Snapshot snapshot : table.snapshots()) {
                if (snapshot.snapshotId() != fromSnapshotId) continue;
                return snapshot.timestampMillis();
            }
        }
        return Long.MAX_VALUE;
    }

    public static Snapshot findLatestSnapshotContainsKey(Table table, String summaryKey) {
        Snapshot latestSnapshot = null;
        for (Snapshot snapshot : table.snapshots()) {
            if (!snapshot.summary().containsKey(summaryKey)) continue;
            latestSnapshot = snapshot;
        }
        return latestSnapshot;
    }

    public static long fetchLatestNonOptimizedSnapshotTime(Table table) {
        Optional<Snapshot> snapshot = IcebergTableUtil.findFirstMatchSnapshot(table, (Predicate<Snapshot>)((Predicate)s -> s.summary().values().stream().noneMatch(AMORO_MAINTAIN_COMMITS::contains)));
        return snapshot.map(Snapshot::timestampMillis).orElse(Long.MAX_VALUE);
    }

    private Set<String> deleteInvalidFilesInFs(SupportsFileSystemOperations fio, String location, long lastTime, Set<String> excludes, Set<PathInfo> directories) {
        if (!fio.exists(location)) {
            return Collections.emptySet();
        }
        HashSet<String> filesToDelete = new HashSet<String>();
        for (PathInfo p : fio.listDirectory(location)) {
            String uriPath = TableFileUtil.getUriPath((String)p.location());
            if (p.isDirectory()) {
                directories.add(p);
                filesToDelete.addAll(this.deleteInvalidFilesInFs(fio, p.location(), lastTime, excludes, directories));
                continue;
            }
            String parentLocation = TableFileUtil.getParent((String)p.location());
            String parentUriPath = TableFileUtil.getUriPath((String)parentLocation);
            if (excludes.contains(uriPath) || excludes.contains(parentUriPath) || p.createdAtMillis() >= lastTime) continue;
            filesToDelete.add(p.location());
        }
        return filesToDelete;
    }

    private void deleteEmptyDirectories(SupportsFileSystemOperations fio, Set<PathInfo> paths, long lastTime, Set<String> excludes) {
        paths.forEach(p -> {
            if (fio.exists(p.location()) && !p.location().endsWith(METADATA_FOLDER_NAME) && !p.location().endsWith(DATA_FOLDER_NAME) && p.createdAtMillis() < lastTime && fio.isEmptyDirectory(p.location())) {
                TableFileUtil.deleteEmptyDirectory((AuthenticatedFileIO)fio, (String)p.location(), (Set)excludes);
            }
        });
    }

    private Set<String> deleteInvalidFilesByPrefix(SupportsPrefixOperations pio, String prefix, long lastTime, Set<String> excludes) {
        HashSet<String> filesToDelete = new HashSet<String>();
        for (FileInfo fileInfo : pio.listPrefix(prefix)) {
            String uriPath = TableFileUtil.getUriPath((String)fileInfo.location());
            if (excludes.contains(uriPath) || fileInfo.createdAtMillis() >= lastTime) continue;
            filesToDelete.add(fileInfo.location());
        }
        return filesToDelete;
    }

    private static Set<String> getValidMetadataFiles(Table internalTable) {
        String tableName = internalTable.name();
        HashSet<String> validFiles = new HashSet<String>();
        Iterable snapshots = internalTable.snapshots();
        int size = Iterables.size((Iterable)snapshots);
        LOG.info("Found {} snapshots to scan for table {}", (Object)size, (Object)tableName);
        for (Snapshot snapshot : snapshots) {
            String manifestListLocation = snapshot.manifestListLocation();
            validFiles.add(TableFileUtil.getUriPath((String)manifestListLocation));
        }
        Set<String> allManifestFiles = IcebergTableUtil.getAllManifestFiles(internalTable);
        allManifestFiles.forEach(f -> validFiles.add(TableFileUtil.getUriPath((String)f)));
        Stream.of(ReachableFileUtil.metadataFileLocations((Table)internalTable, (boolean)false).stream(), ReachableFileUtil.statisticsFilesLocations((Table)internalTable).stream(), Stream.of(ReachableFileUtil.versionHintLocation((Table)internalTable))).reduce(Stream::concat).orElse(Stream.empty()).map(TableFileUtil::getUriPath).forEach(validFiles::add);
        return validFiles;
    }

    private static Pattern getExcludeFileNameRegex(Table table) {
        String latestFlinkJobId = null;
        for (Snapshot snapshot : table.snapshots()) {
            String flinkJobId = (String)snapshot.summary().get(FLINK_JOB_ID);
            if (Strings.isNullOrEmpty((String)flinkJobId)) continue;
            latestFlinkJobId = flinkJobId;
        }
        if (latestFlinkJobId != null) {
            return Pattern.compile(latestFlinkJobId + ".*");
        }
        return null;
    }

    private Set<String> deleteInvalidMetadataFile(SupportsPrefixOperations pio, String location, long lastTime, Set<String> exclude, Pattern excludeRegex) {
        HashSet<String> filesToDelete = new HashSet<String>();
        for (FileInfo fileInfo : pio.listPrefix(location)) {
            String uriPath = TableFileUtil.getUriPath((String)fileInfo.location());
            if (exclude.contains(uriPath) || fileInfo.createdAtMillis() >= lastTime || excludeRegex != null && excludeRegex.matcher(TableFileUtil.getFileName((String)fileInfo.location())).matches()) continue;
            filesToDelete.add(fileInfo.location());
        }
        return filesToDelete;
    }

    CloseableIterable<FileEntry> fileScan(Table table, Expression dataFilter, DataExpirationConfig expirationConfig, long expireTimestamp) {
        TableScan tableScan = (TableScan)((TableScan)table.newScan().filter(dataFilter)).includeColumnStats();
        Snapshot snapshot = IcebergTableUtil.getSnapshot(table, false);
        if (snapshot == null) {
            return CloseableIterable.empty();
        }
        long snapshotId = snapshot.snapshotId();
        CloseableIterable tasks = snapshotId == -1L ? tableScan.planFiles() : tableScan.useSnapshot(snapshotId).planFiles();
        long deleteFileCnt = Long.parseLong(snapshot.summary().getOrDefault("total-delete-files", "0"));
        CloseableIterable dataFiles = CloseableIterable.transform((CloseableIterable)tasks, ContentScanTask::file);
        CloseableIterable hasDeleteTask = deleteFileCnt > 0L ? CloseableIterable.filter((CloseableIterable)tasks, t -> !t.deletes().isEmpty()) : CloseableIterable.empty();
        Set deleteFiles = StreamSupport.stream(hasDeleteTask.spliterator(), true).flatMap(e -> e.deletes().stream()).collect(Collectors.toSet());
        Types.NestedField field = table.schema().findField(expirationConfig.getExpirationField());
        Comparable<?> expireValue = this.getExpireValue(expirationConfig, field, expireTimestamp);
        return CloseableIterable.transform((CloseableIterable)CloseableIterable.withNoopClose((Iterable)Iterables.concat((Iterable)dataFiles, deleteFiles)), contentFile -> {
            Literal<Long> literal = this.getExpireTimestampLiteral((ContentFile<?>)contentFile, field, DateTimeFormatter.ofPattern(expirationConfig.getDateTimePattern(), Locale.getDefault()), expirationConfig.getNumberDateFormat(), expireValue);
            return new FileEntry((ContentFile)contentFile.copyWithoutStats(), literal);
        });
    }

    protected ExpireFiles expiredFileScan(DataExpirationConfig expirationConfig, Expression dataFilter, long expireTimestamp) {
        ConcurrentMap partitionFreshness = Maps.newConcurrentMap();
        ExpireFiles expiredFiles = new ExpireFiles();
        try (CloseableIterable<FileEntry> entries = this.fileScan(this.table, dataFilter, expirationConfig, expireTimestamp);){
            LinkedTransferQueue fileEntries = new LinkedTransferQueue();
            entries.forEach(e -> {
                if (IcebergTableMaintainer.mayExpired(e, partitionFreshness, expireTimestamp)) {
                    fileEntries.add(e);
                }
            });
            fileEntries.parallelStream().filter(e -> IcebergTableMaintainer.willNotRetain(e, expirationConfig, partitionFreshness)).forEach(expiredFiles::addFile);
        }
        catch (IOException e2) {
            throw new RuntimeException(e2);
        }
        return expiredFiles;
    }

    private Comparable<?> getExpireValue(DataExpirationConfig expirationConfig, Types.NestedField field, long expireTimestamp) {
        switch (field.type().typeId()) {
            case TIMESTAMP: {
                return expireTimestamp * 1000L;
            }
            case LONG: {
                if (expirationConfig.getNumberDateFormat().equals(EXPIRE_TIMESTAMP_MS)) {
                    return expireTimestamp;
                }
                if (expirationConfig.getNumberDateFormat().equals(EXPIRE_TIMESTAMP_S)) {
                    return expireTimestamp / 1000L;
                }
                throw new IllegalArgumentException("Number dateformat: " + expirationConfig.getNumberDateFormat());
            }
            case STRING: {
                return LocalDateTime.ofInstant(Instant.ofEpochMilli(expireTimestamp), IcebergTableMaintainer.getDefaultZoneId(field)).format(DateTimeFormatter.ofPattern(expirationConfig.getDateTimePattern(), Locale.getDefault()));
            }
        }
        throw new IllegalArgumentException("Unsupported expiration field type: " + field.type().typeId());
    }

    protected static Expression getDataExpression(Schema schema, DataExpirationConfig expirationConfig, long expireTimestamp) {
        if (expirationConfig.getExpirationLevel().equals((Object)DataExpirationConfig.ExpireLevel.PARTITION)) {
            return Expressions.alwaysTrue();
        }
        Types.NestedField field = schema.findField(expirationConfig.getExpirationField());
        Type.TypeID typeID = field.type().typeId();
        switch (typeID) {
            case TIMESTAMP: {
                return Expressions.lessThanOrEqual((String)field.name(), (Object)(expireTimestamp * 1000L));
            }
            case LONG: {
                if (expirationConfig.getNumberDateFormat().equals(EXPIRE_TIMESTAMP_MS)) {
                    return Expressions.lessThanOrEqual((String)field.name(), (Object)expireTimestamp);
                }
                if (expirationConfig.getNumberDateFormat().equals(EXPIRE_TIMESTAMP_S)) {
                    return Expressions.lessThanOrEqual((String)field.name(), (Object)(expireTimestamp / 1000L));
                }
                return Expressions.alwaysTrue();
            }
            case STRING: {
                String expireDateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(expireTimestamp), IcebergTableMaintainer.getDefaultZoneId(field)).format(DateTimeFormatter.ofPattern(expirationConfig.getDateTimePattern(), Locale.getDefault()));
                return Expressions.lessThanOrEqual((String)field.name(), (Object)expireDateTime);
            }
        }
        return Expressions.alwaysTrue();
    }

    void expireFiles(ExpireFiles expiredFiles, long expireTimestamp) {
        long snapshotId = IcebergTableUtil.getSnapshotId(this.table, false);
        Queue<DataFile> dataFiles = expiredFiles.dataFiles;
        Queue<DeleteFile> deleteFiles = expiredFiles.deleteFiles;
        if (dataFiles.isEmpty() && deleteFiles.isEmpty()) {
            return;
        }
        DeleteFiles delete = this.table.newDelete();
        dataFiles.forEach(arg_0 -> ((DeleteFiles)delete).deleteFile(arg_0));
        delete.set("snapshot.producer", CommitMetaProducer.DATA_EXPIRATION.name());
        delete.commit();
        if (!deleteFiles.isEmpty()) {
            RewriteFiles rewriteFiles = this.table.newRewrite().validateFromSnapshot(snapshotId);
            deleteFiles.forEach(arg_0 -> ((RewriteFiles)rewriteFiles).deleteFile(arg_0));
            rewriteFiles.set("snapshot.producer", CommitMetaProducer.DATA_EXPIRATION.name());
            rewriteFiles.commit();
        }
        LOG.info("Expired files older than {}, {} data files[{}] and {} delete files[{}] for table {}", new Object[]{expireTimestamp, dataFiles.size(), dataFiles.stream().map(ContentFile::path).collect(Collectors.joining(",")), deleteFiles.size(), deleteFiles.stream().map(ContentFile::path).collect(Collectors.joining(",")), this.table.name()});
    }

    static boolean mayExpired(FileEntry fileEntry, Map<StructLike, DataFileFreshness> partitionFreshness, Long expireTimestamp) {
        ContentFile<?> contentFile = fileEntry.getFile();
        StructLike partition = contentFile.partition();
        boolean expired = true;
        if (contentFile.content().equals((Object)FileContent.DATA)) {
            Literal<Long> literal = fileEntry.getTsBound();
            if (partitionFreshness.containsKey(partition)) {
                DataFileFreshness freshness = partitionFreshness.get(partition).incTotalCount();
                if (freshness.latestUpdateMillis <= (Long)literal.value()) {
                    partitionFreshness.put(partition, freshness.updateLatestMillis((Long)literal.value()));
                }
            } else {
                partitionFreshness.putIfAbsent(partition, new DataFileFreshness(fileEntry.getFile().dataSequenceNumber(), (Long)literal.value()).incTotalCount());
            }
            boolean bl = expired = literal.comparator().compare(expireTimestamp, literal.value()) >= 0;
            if (expired) {
                partitionFreshness.computeIfPresent(partition, (k, v) -> v.updateExpiredSeq(fileEntry.getFile().dataSequenceNumber()).incExpiredCount());
            }
        }
        return expired;
    }

    static boolean willNotRetain(FileEntry fileEntry, DataExpirationConfig expirationConfig, Map<StructLike, DataFileFreshness> partitionFreshness) {
        ContentFile<?> contentFile = fileEntry.getFile();
        switch (expirationConfig.getExpirationLevel()) {
            case PARTITION: {
                return partitionFreshness.containsKey(contentFile.partition()) && partitionFreshness.get((Object)contentFile.partition()).expiredDataFileCount == partitionFreshness.get((Object)contentFile.partition()).totalDataFileCount;
            }
            case FILE: {
                if (!contentFile.content().equals((Object)FileContent.DATA)) {
                    long seqUpperBound = partitionFreshness.getOrDefault((Object)contentFile.partition(), (DataFileFreshness)new DataFileFreshness((long)Long.MIN_VALUE, (long)Long.MAX_VALUE)).latestExpiredSeq;
                    return fileEntry.getFile().dataSequenceNumber() <= seqUpperBound;
                }
                return true;
            }
        }
        return false;
    }

    private Literal<Long> getExpireTimestampLiteral(ContentFile<?> contentFile, Types.NestedField field, DateTimeFormatter formatter, String numberDateFormatter, Comparable<?> expireValue) {
        Type type = field.type();
        Object upperBound = Conversions.fromByteBuffer((Type)type, (ByteBuffer)((ByteBuffer)contentFile.upperBounds().get(field.fieldId())));
        Literal literal = Literal.of((long)Long.MAX_VALUE);
        if (null == upperBound) {
            if (this.canBeExpireByPartitionValue(contentFile, field, expireValue)) {
                literal = Literal.of((long)0L);
            }
        } else if (upperBound instanceof Long) {
            switch (type.typeId()) {
                case TIMESTAMP: {
                    literal = Literal.of((long)((Long)upperBound / 1000L));
                    break;
                }
                default: {
                    if (numberDateFormatter.equals(EXPIRE_TIMESTAMP_MS)) {
                        literal = Literal.of((long)((Long)upperBound));
                        break;
                    }
                    if (numberDateFormatter.equals(EXPIRE_TIMESTAMP_S)) {
                        literal = Literal.of((long)((Long)upperBound * 1000L));
                        break;
                    } else {
                        break;
                    }
                }
            }
        } else if (type.typeId().equals((Object)Type.TypeID.STRING)) {
            literal = Literal.of((long)LocalDate.parse(upperBound.toString(), formatter).atStartOfDay().atZone(IcebergTableMaintainer.getDefaultZoneId(field)).toInstant().toEpochMilli());
        }
        return literal;
    }

    private boolean canBeExpireByPartitionValue(ContentFile<?> contentFile, Types.NestedField expireField, Comparable<?> expireValue) {
        PartitionSpec partitionSpec = (PartitionSpec)this.table.specs().get(contentFile.specId());
        int pos = 0;
        ArrayList<Boolean> compareResults = new ArrayList<Boolean>();
        for (PartitionField partitionField : partitionSpec.fields()) {
            if (partitionField.sourceId() == expireField.fieldId()) {
                if (partitionField.transform().isVoid()) {
                    return false;
                }
                Comparable partitionUpperBound = (Comparable)partitionField.transform().bind(expireField.type()).apply(expireValue);
                Comparable filePartitionValue = (Comparable)contentFile.partition().get(pos, partitionUpperBound.getClass());
                int compared = filePartitionValue.compareTo(partitionUpperBound);
                Boolean compareResult = expireField.type() == Types.StringType.get() ? compared <= 0 : compared < 0;
                compareResults.add(compareResult);
            }
            ++pos;
        }
        return !compareResults.isEmpty() && compareResults.stream().allMatch(Boolean::booleanValue);
    }

    public Table getTable() {
        return this.table;
    }

    public static ZoneId getDefaultZoneId(Types.NestedField expireField) {
        Type type = expireField.type();
        if (type.typeId() == Type.TypeID.STRING) {
            return ZoneId.systemDefault();
        }
        return ZoneOffset.UTC;
    }

    private void runWithCondition(boolean condition, Runnable fun) {
        if (condition) {
            fun.run();
        }
    }

    public static class FileEntry {
        private final ContentFile<?> file;
        private final Literal<Long> tsBound;

        FileEntry(ContentFile<?> file, Literal<Long> tsBound) {
            this.file = file;
            this.tsBound = tsBound;
        }

        public ContentFile<?> getFile() {
            return this.file;
        }

        public Literal<Long> getTsBound() {
            return this.tsBound;
        }
    }

    public static class DataFileFreshness {
        long latestExpiredSeq;
        long latestUpdateMillis;
        long expiredDataFileCount;
        long totalDataFileCount;

        DataFileFreshness(long sequenceNumber, long latestUpdateMillis) {
            this.latestExpiredSeq = sequenceNumber;
            this.latestUpdateMillis = latestUpdateMillis;
        }

        DataFileFreshness updateLatestMillis(long ts) {
            this.latestUpdateMillis = ts;
            return this;
        }

        DataFileFreshness updateExpiredSeq(Long seq) {
            this.latestExpiredSeq = seq;
            return this;
        }

        DataFileFreshness incTotalCount() {
            ++this.totalDataFileCount;
            return this;
        }

        DataFileFreshness incExpiredCount() {
            ++this.expiredDataFileCount;
            return this;
        }
    }

    public static class ExpireFiles {
        Queue<DataFile> dataFiles = new LinkedTransferQueue<DataFile>();
        Queue<DeleteFile> deleteFiles = new LinkedTransferQueue<DeleteFile>();

        ExpireFiles() {
        }

        void addFile(FileEntry entry) {
            ContentFile<?> file = entry.getFile();
            switch (file.content()) {
                case DATA: {
                    this.dataFiles.add((DataFile)file.copyWithoutStats());
                    break;
                }
                case EQUALITY_DELETES: 
                case POSITION_DELETES: {
                    this.deleteFiles.add((DeleteFile)file.copyWithoutStats());
                    break;
                }
                default: {
                    throw new IllegalArgumentException(file.content().name() + "cannot be expired");
                }
            }
        }
    }
}

