﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/kinesis/KinesisRequest.h>
#include <aws/kinesis/Kinesis_EXPORTS.h>
#include <aws/kinesis/model/MetricsName.h>

#include <utility>

namespace Aws {
namespace Kinesis {
namespace Model {

/**
 * <p>Represents the input for <a>EnableEnhancedMonitoring</a>.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/EnableEnhancedMonitoringInput">AWS
 * API Reference</a></p>
 */
class EnableEnhancedMonitoringRequest : public KinesisRequest {
 public:
  AWS_KINESIS_API EnableEnhancedMonitoringRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "EnableEnhancedMonitoring"; }

  AWS_KINESIS_API Aws::String SerializePayload() const override;

  AWS_KINESIS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_KINESIS_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p>The name of the stream for which to enable enhanced monitoring.</p>
   */
  inline const Aws::String& GetStreamName() const { return m_streamName; }
  inline bool StreamNameHasBeenSet() const { return m_streamNameHasBeenSet; }
  template <typename StreamNameT = Aws::String>
  void SetStreamName(StreamNameT&& value) {
    m_streamNameHasBeenSet = true;
    m_streamName = std::forward<StreamNameT>(value);
  }
  template <typename StreamNameT = Aws::String>
  EnableEnhancedMonitoringRequest& WithStreamName(StreamNameT&& value) {
    SetStreamName(std::forward<StreamNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>List of shard-level metrics to enable.</p> <p>The following are the valid
   * shard-level metrics. The value "<code>ALL</code>" enables every metric.</p> <ul>
   * <li> <p> <code>IncomingBytes</code> </p> </li> <li> <p>
   * <code>IncomingRecords</code> </p> </li> <li> <p> <code>OutgoingBytes</code> </p>
   * </li> <li> <p> <code>OutgoingRecords</code> </p> </li> <li> <p>
   * <code>WriteProvisionedThroughputExceeded</code> </p> </li> <li> <p>
   * <code>ReadProvisionedThroughputExceeded</code> </p> </li> <li> <p>
   * <code>IteratorAgeMilliseconds</code> </p> </li> <li> <p> <code>ALL</code> </p>
   * </li> </ul> <p>For more information, see <a
   * href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring
   * the Amazon Kinesis Data Streams Service with Amazon CloudWatch</a> in the
   * <i>Amazon Kinesis Data Streams Developer Guide</i>.</p>
   */
  inline const Aws::Vector<MetricsName>& GetShardLevelMetrics() const { return m_shardLevelMetrics; }
  inline bool ShardLevelMetricsHasBeenSet() const { return m_shardLevelMetricsHasBeenSet; }
  template <typename ShardLevelMetricsT = Aws::Vector<MetricsName>>
  void SetShardLevelMetrics(ShardLevelMetricsT&& value) {
    m_shardLevelMetricsHasBeenSet = true;
    m_shardLevelMetrics = std::forward<ShardLevelMetricsT>(value);
  }
  template <typename ShardLevelMetricsT = Aws::Vector<MetricsName>>
  EnableEnhancedMonitoringRequest& WithShardLevelMetrics(ShardLevelMetricsT&& value) {
    SetShardLevelMetrics(std::forward<ShardLevelMetricsT>(value));
    return *this;
  }
  inline EnableEnhancedMonitoringRequest& AddShardLevelMetrics(MetricsName value) {
    m_shardLevelMetricsHasBeenSet = true;
    m_shardLevelMetrics.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the stream.</p>
   */
  inline const Aws::String& GetStreamARN() const { return m_streamARN; }
  inline bool StreamARNHasBeenSet() const { return m_streamARNHasBeenSet; }
  template <typename StreamARNT = Aws::String>
  void SetStreamARN(StreamARNT&& value) {
    m_streamARNHasBeenSet = true;
    m_streamARN = std::forward<StreamARNT>(value);
  }
  template <typename StreamARNT = Aws::String>
  EnableEnhancedMonitoringRequest& WithStreamARN(StreamARNT&& value) {
    SetStreamARN(std::forward<StreamARNT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_streamName;

  Aws::Vector<MetricsName> m_shardLevelMetrics;

  Aws::String m_streamARN;
  bool m_streamNameHasBeenSet = false;
  bool m_shardLevelMetricsHasBeenSet = false;
  bool m_streamARNHasBeenSet = false;
};

}  // namespace Model
}  // namespace Kinesis
}  // namespace Aws
