﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/bedrock-agent/BedrockAgent_EXPORTS.h>
#include <aws/bedrock-agent/model/KnowledgeBasePromptTemplate.h>
#include <aws/bedrock-agent/model/PerformanceConfiguration.h>
#include <aws/bedrock-agent/model/PromptInferenceConfiguration.h>
#include <aws/core/utils/Document.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace BedrockAgent {
namespace Model {

/**
 * <p>Configures how the knowledge base orchestrates the retrieval and generation
 * process, allowing for customization of prompts, inference parameters, and
 * performance settings.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-2023-06-05/KnowledgeBaseOrchestrationConfiguration">AWS
 * API Reference</a></p>
 */
class KnowledgeBaseOrchestrationConfiguration {
 public:
  AWS_BEDROCKAGENT_API KnowledgeBaseOrchestrationConfiguration() = default;
  AWS_BEDROCKAGENT_API KnowledgeBaseOrchestrationConfiguration(Aws::Utils::Json::JsonView jsonValue);
  AWS_BEDROCKAGENT_API KnowledgeBaseOrchestrationConfiguration& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_BEDROCKAGENT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>A custom prompt template for orchestrating the retrieval and generation
   * process.</p>
   */
  inline const KnowledgeBasePromptTemplate& GetPromptTemplate() const { return m_promptTemplate; }
  inline bool PromptTemplateHasBeenSet() const { return m_promptTemplateHasBeenSet; }
  template <typename PromptTemplateT = KnowledgeBasePromptTemplate>
  void SetPromptTemplate(PromptTemplateT&& value) {
    m_promptTemplateHasBeenSet = true;
    m_promptTemplate = std::forward<PromptTemplateT>(value);
  }
  template <typename PromptTemplateT = KnowledgeBasePromptTemplate>
  KnowledgeBaseOrchestrationConfiguration& WithPromptTemplate(PromptTemplateT&& value) {
    SetPromptTemplate(std::forward<PromptTemplateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains inference configurations for the prompt.</p>
   */
  inline const PromptInferenceConfiguration& GetInferenceConfig() const { return m_inferenceConfig; }
  inline bool InferenceConfigHasBeenSet() const { return m_inferenceConfigHasBeenSet; }
  template <typename InferenceConfigT = PromptInferenceConfiguration>
  void SetInferenceConfig(InferenceConfigT&& value) {
    m_inferenceConfigHasBeenSet = true;
    m_inferenceConfig = std::forward<InferenceConfigT>(value);
  }
  template <typename InferenceConfigT = PromptInferenceConfiguration>
  KnowledgeBaseOrchestrationConfiguration& WithInferenceConfig(InferenceConfigT&& value) {
    SetInferenceConfig(std::forward<InferenceConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The additional model-specific request parameters as key-value pairs to be
   * included in the request to the foundation model.</p>
   */
  inline const Aws::Map<Aws::String, Aws::Utils::Document>& GetAdditionalModelRequestFields() const {
    return m_additionalModelRequestFields;
  }
  inline bool AdditionalModelRequestFieldsHasBeenSet() const { return m_additionalModelRequestFieldsHasBeenSet; }
  template <typename AdditionalModelRequestFieldsT = Aws::Map<Aws::String, Aws::Utils::Document>>
  void SetAdditionalModelRequestFields(AdditionalModelRequestFieldsT&& value) {
    m_additionalModelRequestFieldsHasBeenSet = true;
    m_additionalModelRequestFields = std::forward<AdditionalModelRequestFieldsT>(value);
  }
  template <typename AdditionalModelRequestFieldsT = Aws::Map<Aws::String, Aws::Utils::Document>>
  KnowledgeBaseOrchestrationConfiguration& WithAdditionalModelRequestFields(AdditionalModelRequestFieldsT&& value) {
    SetAdditionalModelRequestFields(std::forward<AdditionalModelRequestFieldsT>(value));
    return *this;
  }
  template <typename AdditionalModelRequestFieldsKeyT = Aws::String, typename AdditionalModelRequestFieldsValueT = Aws::Utils::Document>
  KnowledgeBaseOrchestrationConfiguration& AddAdditionalModelRequestFields(AdditionalModelRequestFieldsKeyT&& key,
                                                                           AdditionalModelRequestFieldsValueT&& value) {
    m_additionalModelRequestFieldsHasBeenSet = true;
    m_additionalModelRequestFields.emplace(std::forward<AdditionalModelRequestFieldsKeyT>(key),
                                           std::forward<AdditionalModelRequestFieldsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The performance configuration options for the knowledge base retrieval and
   * generation process.</p>
   */
  inline const PerformanceConfiguration& GetPerformanceConfig() const { return m_performanceConfig; }
  inline bool PerformanceConfigHasBeenSet() const { return m_performanceConfigHasBeenSet; }
  template <typename PerformanceConfigT = PerformanceConfiguration>
  void SetPerformanceConfig(PerformanceConfigT&& value) {
    m_performanceConfigHasBeenSet = true;
    m_performanceConfig = std::forward<PerformanceConfigT>(value);
  }
  template <typename PerformanceConfigT = PerformanceConfiguration>
  KnowledgeBaseOrchestrationConfiguration& WithPerformanceConfig(PerformanceConfigT&& value) {
    SetPerformanceConfig(std::forward<PerformanceConfigT>(value));
    return *this;
  }
  ///@}
 private:
  KnowledgeBasePromptTemplate m_promptTemplate;

  PromptInferenceConfiguration m_inferenceConfig;

  Aws::Map<Aws::String, Aws::Utils::Document> m_additionalModelRequestFields;

  PerformanceConfiguration m_performanceConfig;
  bool m_promptTemplateHasBeenSet = false;
  bool m_inferenceConfigHasBeenSet = false;
  bool m_additionalModelRequestFieldsHasBeenSet = false;
  bool m_performanceConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace BedrockAgent
}  // namespace Aws
