/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.puffin;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import javax.annotation.Nullable;
import org.apache.iceberg.io.InputFile;
import org.apache.iceberg.io.RangeReadable;
import org.apache.iceberg.io.SeekableInputStream;
import org.apache.iceberg.puffin.BlobMetadata;
import org.apache.iceberg.puffin.FileMetadata;
import org.apache.iceberg.puffin.FileMetadataParser;
import org.apache.iceberg.puffin.PuffinCompressionCodec;
import org.apache.iceberg.puffin.PuffinFormat;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.relocated.com.google.common.io.ByteStreams;
import org.apache.iceberg.util.Pair;

public class PuffinReader
implements Closeable {
    private static final byte[] MAGIC = PuffinFormat.getMagic();
    private final long fileSize;
    private final SeekableInputStream input;
    private Integer knownFooterSize;
    private FileMetadata knownFileMetadata;

    PuffinReader(InputFile inputFile, @Nullable Long fileSize, @Nullable Long footerSize) {
        Preconditions.checkNotNull((Object)inputFile, (Object)"inputFile is null");
        this.fileSize = fileSize == null ? inputFile.getLength() : fileSize.longValue();
        this.input = inputFile.newStream();
        if (footerSize != null) {
            Preconditions.checkArgument((0L < footerSize && footerSize <= this.fileSize - (long)MAGIC.length ? 1 : 0) != 0, (String)"Invalid footer size: %s", (Object)footerSize);
            this.knownFooterSize = Math.toIntExact(footerSize);
        }
    }

    public FileMetadata fileMetadata() throws IOException {
        if (this.knownFileMetadata == null) {
            int footerSize = this.footerSize();
            byte[] footer = this.readInput(this.fileSize - (long)footerSize, footerSize);
            PuffinReader.checkMagic(footer, 0);
            int footerStructOffset = footerSize - PuffinFormat.FOOTER_STRUCT_LENGTH;
            PuffinReader.checkMagic(footer, footerStructOffset + 8);
            PuffinCompressionCodec footerCompression = PuffinCompressionCodec.NONE;
            block3: for (PuffinFormat.Flag flag : this.decodeFlags(footer, footerStructOffset)) {
                switch (flag) {
                    case FOOTER_PAYLOAD_COMPRESSED: {
                        footerCompression = PuffinFormat.FOOTER_COMPRESSION_CODEC;
                        continue block3;
                    }
                }
                throw new IllegalStateException("Unsupported flag: " + String.valueOf((Object)flag));
            }
            int footerPayloadSize = PuffinFormat.readIntegerLittleEndian(footer, footerStructOffset + 0);
            Preconditions.checkState((footerSize == PuffinFormat.FOOTER_START_MAGIC_LENGTH + footerPayloadSize + PuffinFormat.FOOTER_STRUCT_LENGTH ? 1 : 0) != 0, (String)"Unexpected footer payload size value %s for footer size %s", (int)footerPayloadSize, (int)footerSize);
            ByteBuffer footerPayload = ByteBuffer.wrap(footer, 4, footerPayloadSize);
            ByteBuffer footerJson = PuffinFormat.decompress(footerCompression, footerPayload);
            this.knownFileMetadata = PuffinReader.parseFileMetadata(footerJson);
        }
        return this.knownFileMetadata;
    }

    private Set<PuffinFormat.Flag> decodeFlags(byte[] footer, int footerStructOffset) {
        EnumSet<PuffinFormat.Flag> flags = EnumSet.noneOf(PuffinFormat.Flag.class);
        for (int byteNumber = 0; byteNumber < 4; ++byteNumber) {
            int flagByte = Byte.toUnsignedInt(footer[footerStructOffset + 4 + byteNumber]);
            int bitNumber = 0;
            while (flagByte != 0) {
                if ((flagByte & 1) != 0) {
                    PuffinFormat.Flag flag = PuffinFormat.Flag.fromBit(byteNumber, bitNumber);
                    Preconditions.checkState((flag != null ? 1 : 0) != 0, (String)"Unknown flag byte %s and bit %s set", (int)byteNumber, (int)bitNumber);
                    flags.add(flag);
                }
                flagByte >>= 1;
                ++bitNumber;
            }
        }
        return flags;
    }

    public Iterable<Pair<BlobMetadata, ByteBuffer>> readAll(List<BlobMetadata> blobs) {
        if (blobs.isEmpty()) {
            return ImmutableList.of();
        }
        return () -> blobs.stream().sorted(Comparator.comparingLong(BlobMetadata::offset)).map(blobMetadata -> {
            try {
                this.input.seek(blobMetadata.offset());
                byte[] bytes = new byte[Math.toIntExact(blobMetadata.length())];
                ByteStreams.readFully((InputStream)this.input, (byte[])bytes);
                ByteBuffer rawData = ByteBuffer.wrap(bytes);
                PuffinCompressionCodec codec = PuffinCompressionCodec.forName(blobMetadata.compressionCodec());
                ByteBuffer data = PuffinFormat.decompress(codec, rawData);
                return Pair.of(blobMetadata, data);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }).iterator();
    }

    private static void checkMagic(byte[] data, int offset) {
        byte[] read = Arrays.copyOfRange(data, offset, offset + MAGIC.length);
        if (!Arrays.equals(read, MAGIC)) {
            throw new IllegalStateException(String.format("Invalid file: expected magic at offset %s: %s, but got %s", offset, Arrays.toString(MAGIC), Arrays.toString(read)));
        }
    }

    private int footerSize() throws IOException {
        if (this.knownFooterSize == null) {
            Preconditions.checkState((this.fileSize >= (long)PuffinFormat.FOOTER_STRUCT_LENGTH ? 1 : 0) != 0, (String)"Invalid file: file length %s is less tha minimal length of the footer tail %s", (long)this.fileSize, (int)PuffinFormat.FOOTER_STRUCT_LENGTH);
            byte[] footerStruct = this.readInput(this.fileSize - (long)PuffinFormat.FOOTER_STRUCT_LENGTH, PuffinFormat.FOOTER_STRUCT_LENGTH);
            PuffinReader.checkMagic(footerStruct, 8);
            int footerPayloadSize = PuffinFormat.readIntegerLittleEndian(footerStruct, 0);
            this.knownFooterSize = PuffinFormat.FOOTER_START_MAGIC_LENGTH + footerPayloadSize + PuffinFormat.FOOTER_STRUCT_LENGTH;
        }
        return this.knownFooterSize;
    }

    private byte[] readInput(long offset, int length) throws IOException {
        byte[] data = new byte[length];
        if (this.input instanceof RangeReadable) {
            ((RangeReadable)this.input).readFully(offset, data);
        } else {
            this.input.seek(offset);
            ByteStreams.readFully((InputStream)this.input, (byte[])data);
        }
        return data;
    }

    private static FileMetadata parseFileMetadata(ByteBuffer data) {
        String footerJson = StandardCharsets.UTF_8.decode(data).toString();
        return FileMetadataParser.fromJson(footerJson);
    }

    @Override
    public void close() throws IOException {
        this.input.close();
        this.knownFooterSize = null;
        this.knownFileMetadata = null;
    }
}

