# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Unit tests for DebianPipeline work request plugin."""

from typing import ClassVar, cast, override

from debusine.db.models import WorkRequest
from debusine.db.playground import scenarios
from debusine.test.django import TestCase
from debusine.web.views.debian_pipeline import (
    DebianPipelineViewWorkRequestPlugin,
)
from debusine.web.views.qa import QATab
from debusine.web.views.work_request import Tab


class DebianPipelineViewWorkRequestPluginTests(TestCase):
    """Tests for :py:class:`DebianPipelineViewWorkRequestPlugin`."""

    scenario = scenarios.UIRegressionTesting(simulate=False)

    pipeline_without_qa: ClassVar[WorkRequest]

    @override
    @classmethod
    def setUpTestData(cls) -> None:
        super().setUpTestData()
        cls.pipeline_without_qa = cls.playground.create_workflow(
            cls.scenario.template_upload_to_sid,
            task_data={
                "source_artifact": cls.scenario.new_source.pk,
                "architectures": ["all", "amd64", "i386"],
                "enable_regression_tracking": True,
            },
            workspace=cls.scenario.workspace,
        )

    def plugin(
        self, work_request: WorkRequest | None = None
    ) -> DebianPipelineViewWorkRequestPlugin:
        if work_request is None:
            work_request = self.scenario.workflow_debian_pipeline
        return DebianPipelineViewWorkRequestPlugin(work_request)

    def test_enabled(self) -> None:
        plugin = self.plugin()
        self.assertTrue(plugin.is_enabled())

    def test_disabled_no_qa(self) -> None:
        plugin = self.plugin(self.pipeline_without_qa)
        self.assertFalse(plugin.is_enabled())

    def test_disabled_no_qa_data(self) -> None:
        self.scenario.workflow_qa.output_data = None
        self.scenario.workflow_qa.save()
        plugin = self.plugin()
        self.assertFalse(plugin.is_enabled())

    def test_add_tabs(self) -> None:
        plugin = self.plugin()

        tabs: list[Tab | int] = [1, 2, 3]
        plugin.add_tabs(cast(list[Tab], tabs))
        self.assertEqual(len(tabs), 4)
        self.assertEqual(tabs[0], 1)
        self.assertEqual(tabs[2:], [2, 3])
        tab = tabs[1]
        assert isinstance(tab, QATab)
        self.assertEqual(tab.work_request, self.scenario.workflow_qa)

    def test_add_tabs_disabled(self) -> None:
        plugin = self.plugin(self.pipeline_without_qa)

        tabs: list[int] = [1, 2, 3]
        plugin.add_tabs(cast(list[Tab], tabs))
        self.assertEqual(tabs, [1, 2, 3])

    def test_get_context_data(self) -> None:
        plugin = self.plugin()
        self.assertEqual(plugin.get_context_data(), {})
